<?php
/**
 * Laposta Campaigns Settings Class
 *
 * @package Campaigns_Lite_For_Laposta
 */

if (!defined('ABSPATH')) {
  exit;
}

class Campaigns_Lite_For_Laposta_Settings
{
  /**
   * Option group name
   */
  private $option_group = 'campaigns_lite_for_laposta_options';

  /**
   * Option name for API key
   */
  private $api_key_option = 'campaigns_lite_for_laposta_api_key';

  /**
   * API instance
   */
  private $api;

  /**
   * Constructor
   */
  public function __construct()
  {
    // Initialize API
    require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-api.php';
    $this->api = new Campaigns_Lite_For_Laposta_API();

    add_action('admin_menu', array($this, 'add_settings_page'));
    add_action('admin_init', array($this, 'register_settings'));
    add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
    // Filter for plugin links is now handled in the main plugin file

    // Add admin notice for settings saved
    add_action('admin_init', array($this, 'check_settings_saved'));
    add_action('admin_notices', array($this, 'display_settings_notices'));

    // Help tabs are now added on page load
  }

  /**
   * Check if settings were just saved to show admin notice
   *
   * @since 1.0.0
   */
  public function check_settings_saved()
  {
    // Check if we're on our settings page
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for page identification only, not processing form data
    if (!isset($_GET['page']) || $_GET['page'] !== 'campaigns-lite-for-laposta-settings') {
      return;
    }

    // Notice handling is done through WordPress settings_errors() API
  }

  /**
   * Display admin notices for settings (removed - using WordPress settings_errors() instead)
   *
   * @since 1.0.0
   */
  public function display_settings_notices()
  {
    // Notices are now handled by WordPress settings_errors() API
    // See render_settings_page() method where settings_errors() is called
  }

  /**
   * Add settings page to the admin menu
   */
  public function add_settings_page()
  {
    // Add main menu item
    $main_hook = add_menu_page(
      __('Campaigns Lite for Laposta', 'campaigns-lite-for-laposta'),
      __('Laposta Camp.', 'campaigns-lite-for-laposta'),
      'manage_options',
      'campaigns-lite-for-laposta-settings',
      array($this, 'render_settings_page'),
      'dashicons-email',
      66
    );

    // Add action to add help tabs when this page loads
    add_action('load-' . $main_hook, array($this, 'setup_help_tabs'));

    // Add Settings submenu
    $settings_hook = add_submenu_page(
      'campaigns-lite-for-laposta-settings',
      __('Campaigns Lite for Laposta', 'campaigns-lite-for-laposta'),
      __('Settings', 'campaigns-lite-for-laposta'),
      'manage_options',
      'campaigns-lite-for-laposta-settings',  // Keep base page
      array($this, 'render_settings_page')
    );

    // Add action to add help tabs when this page loads
    add_action('load-' . $settings_hook, array($this, 'setup_help_tabs'));

    // Add Campaigns submenu
    $campaigns_hook = add_submenu_page(
      'campaigns-lite-for-laposta-settings',
      __('Campaigns Lite for Laposta - Campaigns', 'campaigns-lite-for-laposta'),
      __('Campaigns', 'campaigns-lite-for-laposta'),
      'manage_options',
      'campaigns-lite-for-laposta-settings&tab=campaigns',  // Use base page with tab
      array($this, 'render_settings_page')
    );

    // Add action to add help tabs when this page loads
    add_action('load-' . $campaigns_hook, array($this, 'setup_help_tabs'));

    // Add Shortcode submenu
    $shortcode_hook = add_submenu_page(
      'campaigns-lite-for-laposta-settings',
      __('Campaigns Lite for Laposta - Shortcode', 'campaigns-lite-for-laposta'),
      __('Shortcode', 'campaigns-lite-for-laposta'),
      'manage_options',
      'campaigns-lite-for-laposta-settings&tab=shortcode',
      array($this, 'render_settings_page')
    );

    // Add action to add help tabs when this page loads
    add_action('load-' . $shortcode_hook, array($this, 'setup_help_tabs'));

    // Add More Plugins submenu
    $more_plugins_hook = add_submenu_page(
      'campaigns-lite-for-laposta-settings',
      __('More Plugins', 'campaigns-lite-for-laposta'),
      __('More Plugins', 'campaigns-lite-for-laposta'),
      'manage_options',
      'campaigns-lite-for-laposta-settings&tab=more-plugins',
      array($this, 'render_settings_page')
    );

    add_action('load-' . $more_plugins_hook, array($this, 'setup_help_tabs'));
  }

  // Settings link in plugin listing is now handled in the main plugin file

  /**
   * Enqueue admin scripts and styles with cache busting
   */
  public function enqueue_admin_styles($hook)
  {
    // Load menu icon styles on all admin pages
    wp_enqueue_style(
      'laposta-campaigns-lite-menu',
      CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/css/campaigns-lite-for-laposta-menu.css',
      array(),
      CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION
    );

    // Check if we're on any of our plugin pages
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for page identification to conditionally load scripts
    $current_page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : '';
    if (!in_array($current_page, array('campaigns-lite-for-laposta-settings', 'campaigns-lite-for-laposta'))) {
      return;
    }

    // Load admin styles
    wp_enqueue_style(
      'laposta-campaigns-admin',
      CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/css/campaigns-lite-for-laposta-settings.css',
      array(),
      CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION
    );

    // Load admin scripts
    wp_enqueue_script(
      'laposta-campaigns-admin',
      CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/js/campaigns-lite-for-laposta-settings.js',
      array('jquery'),
      CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION,
      true
    );

    wp_localize_script('laposta-campaigns-admin', 'campaignsLiteForLaposta', array(
      'show' => __('Show', 'campaigns-lite-for-laposta'),
      'hide' => __('Hide', 'campaigns-lite-for-laposta'),
      'copied' => __('Copied!', 'campaigns-lite-for-laposta'),
      'copy_to_clipboard' => __('Copy to Clipboard', 'campaigns-lite-for-laposta'),
      'loading' => __('Loading...', 'campaigns-lite-for-laposta'),
      'clearing' => __('Clearing...', 'campaigns-lite-for-laposta'),
      'nonce' => wp_create_nonce('campaigns_lite_for_laposta_ajax'),
      'ajaxurl' => admin_url('admin-ajax.php'),
    ));
  }

  /**
   * AJAX handler for fetching campaign details
   * 
   * Note: Since we now get all campaign data including screenshots and web URLs in a single API call,
   * this method is less important, but we keep it for compatibility and potential manual refresh needs
   */
  public function ajax_fetch_campaign_details()
  {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'campaigns_lite_for_laposta_ajax')) {
      wp_send_json_error(array('message' => esc_html__('Security check failed', 'campaigns-lite-for-laposta')));
    }

    // Verify user capabilities
    if (!current_user_can('manage_options')) {
      wp_send_json_error(array('message' => __('Permission denied', 'campaigns-lite-for-laposta')));
    }

    // Get campaign ID
    $campaign_id = isset($_POST['campaign_id']) ? sanitize_text_field(wp_unslash($_POST['campaign_id'])) : '';
    if (empty($campaign_id)) {
      wp_send_json_error(array('message' => __('Campaign ID is required', 'campaigns-lite-for-laposta')));
    }

    // Make sure API is initialized
    require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-api.php';
    if (!isset($this->api) || !is_object($this->api)) {
      $this->api = new Campaigns_Lite_For_Laposta_API();
    }

    // Clear the specific campaign cache
    $cache_key = 'laposta_campaign_details_' . $campaign_id;
    delete_transient($cache_key);

    // Also clear the all campaigns cache to ensure consistency
    $actual_api_key = self::get_actual_api_key();
    if ($actual_api_key) {
      $all_campaigns_cache_key = 'laposta_all_campaigns_' . md5($actual_api_key);
      delete_transient($all_campaigns_cache_key);
    }

    // Fetch details
    $details = $this->api->get_campaign_details($campaign_id);

    if (is_wp_error($details)) {
      wp_send_json_error(array('message' => $details->get_error_message()));
    }

    if (empty($details) || !isset($details['campaign'])) {
      wp_send_json_error(array('message' => __('No campaign details found', 'campaigns-lite-for-laposta')));
    }

    // Success - the page will be reloaded to show updated data
    wp_send_json_success(array('message' => __('Campaign details updated', 'campaigns-lite-for-laposta')));
  }

  /**
   * AJAX handler for clearing campaign cache
   */
  public function ajax_clear_campaign_cache()
  {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'campaigns_lite_for_laposta_ajax')) {
      wp_send_json_error(array('message' => __('Security check failed', 'campaigns-lite-for-laposta')));
    }

    // Verify user capabilities
    if (!current_user_can('manage_options')) {
      wp_send_json_error(array('message' => __('Permission denied', 'campaigns-lite-for-laposta')));
    }

    // Clear all campaign caches
    $this->clear_all_caches();

    // Success
    wp_send_json_success(array('message' => __('All campaign caches cleared', 'campaigns-lite-for-laposta')));
  }

  // The refresh all campaign details AJAX handler has been removed since it's no longer needed

  /**
   * Register plugin settings
   */
  public function register_settings()
  {
    // Register API key setting
    register_setting(
      $this->option_group,
      $this->api_key_option,
      array(
        'type' => 'string',
        'sanitize_callback' => array($this, 'sanitize_api_key'),
        'default' => ''
      )
    );


    // Add API settings section
    add_settings_section(
      'campaigns_lite_for_laposta_main_section',
      __('API Settings', 'campaigns-lite-for-laposta'),
      array($this, 'render_settings_section'),
      'campaigns-lite-for-laposta-settings'
    );

    // Add API key field
    add_settings_field(
      $this->api_key_option,
      __('API Key', 'campaigns-lite-for-laposta'),
      array($this, 'render_api_key_field'),
      'campaigns-lite-for-laposta-settings',
      'campaigns_lite_for_laposta_main_section'
    );
  }

  /**
   * Render settings section
   */
  public function render_settings_section()
  {
    echo '<p>' . esc_html__('Enter your Laposta API key:', 'campaigns-lite-for-laposta') . '</p>';
  }

  /**
   * Render API key field
   */
  public function render_api_key_field()
  {
    $current_api_key = get_option($this->api_key_option, '');
    ?>
    <div class="api-key-input-wrapper">
      <input
        type="password"
        id="campaigns_lite_for_laposta_api_key"
        name="<?php echo esc_attr($this->api_key_option); ?>"
        value="<?php echo esc_attr($current_api_key); ?>"
        class="regular-text"
        placeholder="<?php esc_attr_e('Enter your Laposta API key', 'campaigns-lite-for-laposta'); ?>"
      />
      <button type="button" class="button button-secondary toggle-api-key"
        data-show="<?php esc_attr_e('Show', 'campaigns-lite-for-laposta'); ?>"
        data-hide="<?php esc_attr_e('Hide', 'campaigns-lite-for-laposta'); ?>">
        <?php esc_html_e('Show', 'campaigns-lite-for-laposta'); ?>
      </button>
    </div>
    <p class="description">
      <?php
      printf(
        // Translators: %1$s is the opening HTML link tag, %2$s is the closing HTML link tag
        esc_html__('%1$sHow do I get an API key?%2$s', 'campaigns-lite-for-laposta'),
        '<a href="https://docs.laposta.nl/article/349-hoe-kom-ik-aan-een-api-sleutel" target="_blank" rel="noopener noreferrer">',
        '</a>'
      );
      ?>
    </p>
    <?php
  }

  /**
   * Sanitize API key
   *
   * @param string $value The unsanitized value
   * @return string The sanitized value
   */
  public function sanitize_api_key($value)
  {
    $old_value = get_option($this->api_key_option);

    // Sanitize the input
    $value = sanitize_text_field($value);

    // Allow empty values - warning will be shown on the settings page
    if ($value === '' || $value === null) {
      return '';
    }

    // If the key has changed, validate it by making a test API call
    if ($value !== $old_value) {
      // Test the API key with a simple API call
      require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-api.php';
      $test_response = Campaigns_Lite_For_Laposta_API_Utils::get($value, 'list');

      if (is_wp_error($test_response)) {
        // Translate common API error messages
        $error_message = $test_response->get_error_message();

        // Check for common error messages and translate them
        if (stripos($error_message, 'No valid API-key provided') !== false || stripos($error_message, 'No valid API key provided.') !== false) {
          $error_message = __('No valid API key provided.', 'campaigns-lite-for-laposta');
        } elseif (stripos($error_message, 'Invalid API key') !== false) {
          $error_message = __('Invalid API key.', 'campaigns-lite-for-laposta');
        } elseif (stripos($error_message, 'API key not found') !== false) {
          $error_message = __('API key not found.', 'campaigns-lite-for-laposta');
        }

        add_settings_error(
          $this->api_key_option,
          'invalid_api_key',
          $error_message
        );
        return $old_value; // Keep the old value if validation fails
      }

      // Clear all caches if validation successful (pass both old and new keys)
      $this->clear_all_caches($old_value, $value);

      // Use a custom transient instead of add_settings_error to avoid duplicate display
      set_transient('campaigns_lite_for_laposta_api_validated', true, 30);
    }

    // Return the sanitized API key
    return $value;
  }

  /**
   * Get the actual API key
   *
   * @return string The actual API key or empty string if not found
   */
  public static function get_actual_api_key()
  {
    // Return the stored API key directly
    return get_option('campaigns_lite_for_laposta_api_key', '');
  }

  /**
   * Track a transient key for later deletion
   *
   * @param string $key The transient key to track
   */
  private function track_transient($key)
  {
    $transient_keys = get_option('campaigns_lite_for_laposta_transient_keys', array());
    
    // Ensure we have an array (in case the option was corrupted)
    if (!is_array($transient_keys)) {
      $transient_keys = array();
    }
    
    if (!in_array($key, $transient_keys)) {
      $transient_keys[] = $key;
      update_option('campaigns_lite_for_laposta_transient_keys', $transient_keys);
    }
  }
  
  /**
   * Clear all plugin caches when API key changes
   */
  private function clear_all_caches($old_api_key = null, $new_api_key = null)
  {
    // Store a list of known transient keys in an option
    $transient_keys = get_option('campaigns_lite_for_laposta_transient_keys', array());

    // Ensure we have an array (in case the option was corrupted)
    if (!is_array($transient_keys)) {
      $transient_keys = array();
    }

    // Delete all stored transients
    foreach ($transient_keys as $key) {
      delete_transient($key);
    }

    // Clear the transient keys list
    update_option('campaigns_lite_for_laposta_transient_keys', array());

    // Collect API keys to clear caches for
    $keys_to_clear = array();

    if ($old_api_key) {
      $keys_to_clear[] = $old_api_key;
    }
    if ($new_api_key) {
      $keys_to_clear[] = $new_api_key;
    }

    // If no keys provided, use current key
    if (empty($keys_to_clear)) {
      $api_key = self::get_actual_api_key();
      if ($api_key) {
        $keys_to_clear[] = $api_key;
      }
    }

    // Clear caches for all relevant API keys
    foreach ($keys_to_clear as $api_key) {
      if ($api_key) {
        $api_key_hash = md5($api_key);

        // Delete cache key patterns (using 'lite' prefix to avoid conflicts with full version)
        // Pattern 1: Used by API Utils class get_all_campaigns()
        delete_transient('campaigns_lite_campaigns_all_' . md5($api_key . serialize(array())));

        // Pattern 2: Used by API class get_campaigns() - THIS IS WHAT THE SHORTCODE USES
        delete_transient('campaigns_lite_all_campaigns_' . $api_key_hash);

        // Pattern 3: Campaign list cache (used in admin)
        delete_transient('campaigns_lite_for_laposta_list_' . $api_key_hash);
      }
    }
  }

  /**
   * Render settings page
   */
  public function render_settings_page()
  {
    // Check user capabilities
    if (!current_user_can('manage_options')) {
      return;
    }

    // Display success message from our custom transient
    if (get_transient('campaigns_lite_for_laposta_api_validated')) {
      ?>
      <div class="notice notice-success is-dismissible">
        <p><?php esc_html_e('API key validated successfully!', 'campaigns-lite-for-laposta'); ?></p>
      </div>
      <?php
      delete_transient('campaigns_lite_for_laposta_api_validated');
    }

    // Display any error messages (errors still use WordPress settings_errors)
    $settings_errors = get_settings_errors($this->api_key_option);
    if (!empty($settings_errors)) {
      foreach ($settings_errors as $error) {
        if ($error['type'] === 'error') {
          printf(
            '<div class="notice notice-error is-dismissible"><p>%s</p></div>',
            esc_html($error['message'])
          );
        }
      }
    }
    ?>
    <div class="wrap">

      <div class="laposta-campaigns-header">
        <h1>
          <img src="<?php echo esc_url(CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/images/icon.svg'); ?>" alt="Laposta"
            class="laposta-logo" style="width: 32px; height: 32px; vertical-align: middle; margin-right: 8px;">
          <?php echo esc_html(get_admin_page_title()); ?>
        </h1>
      </div>

      <h2 class="nav-tab-wrapper">
        <?php
        // Use hardcoded page slug for navigation links
        $page_param = 'campaigns-lite-for-laposta-settings';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for tab navigation URL building and active state, not processing form data
        $tab_param = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : '';
        ?>
        <a href="<?php echo esc_url(add_query_arg(array('page' => $page_param, 'tab' => 'settings'))); ?>"
          class="nav-tab <?php echo empty($tab_param) || $tab_param === 'settings' ? 'nav-tab-active' : ''; ?>">
          <?php esc_html_e('Settings', 'campaigns-lite-for-laposta'); ?>
        </a>
        <a href="<?php echo esc_url(add_query_arg(array('page' => $page_param, 'tab' => 'campaigns'))); ?>"
          class="nav-tab <?php echo $tab_param === 'campaigns' ? 'nav-tab-active' : ''; ?>">
          <?php esc_html_e('Campaigns', 'campaigns-lite-for-laposta'); ?>
        </a>
        <a href="<?php echo esc_url(add_query_arg(array('page' => $page_param, 'tab' => 'shortcode'))); ?>"
          class="nav-tab <?php echo $tab_param === 'shortcode' ? 'nav-tab-active' : ''; ?>">
          <?php esc_html_e('Shortcode', 'campaigns-lite-for-laposta'); ?>
        </a>
        <a href="<?php echo esc_url(add_query_arg(array('page' => $page_param, 'tab' => 'more-plugins'))); ?>"
          class="nav-tab <?php echo $tab_param === 'more-plugins' ? 'nav-tab-active' : ''; ?>">
          <?php esc_html_e('More Plugins', 'campaigns-lite-for-laposta'); ?>
        </a>
      </h2>

      <?php
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for tab identification to render correct content
      $current_page = sanitize_key(wp_unslash($_GET['page'] ?? ''));
      $default_tab = ($current_page === 'campaigns-lite-for-laposta-settings') ? 'settings' : 'campaigns';
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for tab identification to render correct content
      $tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : $default_tab;

      if ($tab === 'settings') {
        $this->display_instructions();
        ?>
        <form action="options.php" method="post">
          <?php
          settings_fields($this->option_group);
          do_settings_sections('campaigns-lite-for-laposta-settings');
          submit_button(__('Save Settings', 'campaigns-lite-for-laposta'));
          ?>
        </form>
        <?php
      } elseif ($tab === 'campaigns') {
        $api_key_index = get_option($this->api_key_option);
        if ($api_key_index === '' || $api_key_index === null || $api_key_index === false) {
          echo '<div class="notice notice-warning"><p>' .
            sprintf(
              /* translators: %s: settings tab link */
              esc_html__('Please configure your API key in the %s tab first.', 'campaigns-lite-for-laposta'),
              '<a href="' . esc_url(admin_url('admin.php?page=campaigns-lite-for-laposta-settings&tab=settings')) . '">' . esc_html__('Settings', 'campaigns-lite-for-laposta') . '</a>'
            ) .
            '</p></div>';
        } else {
          $this->display_campaigns();
        }
      } elseif ($tab === 'shortcode') {
        $this->display_shortcode_settings();
      } elseif ($tab === 'more-plugins') {
        $this->display_more_plugins();
      }
      ?>
    </div>
    <?php
  }

  /**
   * Fetch account information from Laposta API
   *
   * @return array|WP_Error Array of account info or WP_Error on failure
   */
  private function fetch_account_info()
  {
    $api_key = self::get_actual_api_key();
    if (empty($api_key)) {
      // API key is not set
      return new WP_Error('no_api_key', __('API key is not set', 'campaigns-lite-for-laposta'));
    }

    // Use centralized endpoint if available, otherwise use default
    $endpoint = defined('LAPOSTA_API_ENDPOINT') ? LAPOSTA_API_ENDPOINT : 'https://api.laposta.org/v2/';
    $url = $endpoint . 'auth';
    $args = array(
      'headers' => array(
        'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
      ),
      'timeout' => 15, // Set a 15-second timeout
    );

    // Attempting to fetch account info
    $response = wp_remote_get($url, $args);

    if (is_wp_error($response)) {
      // Error in wp_remote_get call
      return $response;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);

    // API Response Code received
    // API Response Body received

    if ($response_code === 429) {
      // Too many requests (429)
      return new WP_Error('too_many_requests', __('Too many requests. Please try again later.', 'campaigns-lite-for-laposta'));
    }

    $data = json_decode($body, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
      // JSON decode error occurred
      return new WP_Error('json_decode_error', __('Error decoding API response', 'campaigns-lite-for-laposta'));
    }

    if (
      isset($data['authenticated']) && $data['authenticated'] === true &&
      isset($data['account_number']) && isset($data['account_name'])
    ) {
      // Successfully retrieved account info
      return array(
        'name' => $data['account_name'],
        'number' => $data['account_number']
      );
    }

    // Unexpected API response structure
    return new WP_Error('unexpected_response', __('Unexpected API response', 'campaigns-lite-for-laposta'));
  }

  /**
   * Process filtering for campaigns based on shortcode parameters
   * 
   * @param array $campaigns List of campaign data
   * @param array $atts Shortcode attributes
   * @return array Filtered campaign list
   */
  public function filter_campaigns($campaigns, $atts)
  {
    // Default parameters
    $defaults = array(
      'include' => '',
      'exclude' => '',
      'include_name' => '',
      'exclude_name' => '',
    );

    // Parse attributes
    $atts = shortcode_atts($defaults, $atts);

    // Prepare filter keywords
    // Note: These 'exclude' parameters are for filtering Laposta API campaigns, not WordPress get_posts()
    // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn
    $include_subject = !empty($atts['include']) ? array_map('trim', explode(',', strtolower($atts['include']))) : array();
    $exclude_subject = !empty($atts['exclude']) ? array_map('trim', explode(',', strtolower($atts['exclude']))) : array();
    $include_name = !empty($atts['include_name']) ? array_map('trim', explode(',', strtolower($atts['include_name']))) : array();
    $exclude_name = !empty($atts['exclude_name']) ? array_map('trim', explode(',', strtolower($atts['exclude_name']))) : array();

    // Return all campaigns if no filters are set
    if (empty($include_subject) && empty($exclude_subject) && empty($include_name) && empty($exclude_name)) {
      return $campaigns;
    }

    $filtered_campaigns = array();

    foreach ($campaigns as $item) {
      if (!isset($item['campaign'])) {
        continue;
      }

      $campaign = $item['campaign'];
      $subject = isset($campaign['subject']) ? strtolower($campaign['subject']) : '';
      $name = isset($campaign['name']) ? strtolower($campaign['name']) : '';

      // Skip if subject contains excluded keywords
      if (!empty($exclude_subject)) {
        $exclude_match = false;
        foreach ($exclude_subject as $keyword) {
          if (!empty($keyword) && strpos($subject, $keyword) !== false) {
            $exclude_match = true;
            break;
          }
        }
        if ($exclude_match) {
          continue;
        }
      }

      // Skip if name contains excluded keywords
      if (!empty($exclude_name)) {
        $exclude_match = false;
        foreach ($exclude_name as $keyword) {
          if (!empty($keyword) && strpos($name, $keyword) !== false) {
            $exclude_match = true;
            break;
          }
        }
        if ($exclude_match) {
          continue;
        }
      }

      // Skip if include_subject is set and subject doesn't contain any included keywords
      if (!empty($include_subject)) {
        $include_match = false;
        foreach ($include_subject as $keyword) {
          if (!empty($keyword) && strpos($subject, $keyword) !== false) {
            $include_match = true;
            break;
          }
        }
        if (!$include_match) {
          continue;
        }
      }

      // Skip if include_name is set and name doesn't contain any included keywords
      if (!empty($include_name)) {
        $include_match = false;
        foreach ($include_name as $keyword) {
          if (!empty($keyword) && strpos($name, $keyword) !== false) {
            $include_match = true;
            break;
          }
        }
        if (!$include_match) {
          continue;
        }
      }

      // If we got here, the campaign passed all filters
      $filtered_campaigns[] = $item;
    }

    return $filtered_campaigns;
  }

  /**
   * Display more plugins by Miix
   */
  private function display_more_plugins()
  {
    ?>
    <div class="wrap">
      <div class="laposta-more-plugins-header">
        <svg xmlns="http://www.w3.org/2000/svg" width="60" height="60" viewBox="0 0 320 320" aria-label="Miix.dev">
          <path fill="#14b8a6" d="M221.308,203.3851h-17.8934l29.4023-43.8682-28.559-42.9021h17.7758l19.7617,30.5498,19.5853-30.5498h17.8934l-28.5589,42.9021,50.5301,75.6937c11.9642-22.4218,18.7546-48.0216,18.7546-75.2106,0-88.3655-71.6344-160-159.9999-160S0,71.6345,0,160s71.6345,160,160,160c55.0378,0,103.5807-27.7927,132.3716-70.1056l-50.5761-78.0251-20.4875,31.5158h0ZM126.1672,203.3851h-13.5587v-55.6173l-26.6318,55.6173h-5.7863l-26.5729-55.6173v55.6173h-13.6175v-86.7703h13.0781l30.0055,61.4625,30.0104-61.4625h13.0732s0,86.7703,0,86.7703ZM158.7618,203.3851h-14.5198v-86.7703h14.5198v86.7703ZM192.5677,203.3851h-14.5198v-86.7703h14.5198v86.7703Z"></path>
        </svg>
        <div>
          <h2><?php esc_html_e('More Plugins by Miix', 'campaigns-lite-for-laposta'); ?></h2>
          <p><?php esc_html_e('Check out my other WordPress plugins for Laposta integration:', 'campaigns-lite-for-laposta'); ?></p>
        </div>
      </div>

      <div class="laposta-plugins-grid">

        <!-- Laposta API Plugin -->
        <div class="laposta-plugin-card">
          <h3><?php esc_html_e('Laposta API', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('General Laposta API management.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-api" target="_blank" class="button button-primary">
            <?php esc_html_e('View Plugin', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

        <!-- Laposta Contact Form 7 Plugin -->
        <div class="laposta-plugin-card">
          <h3><?php esc_html_e('Laposta for Contact Form 7', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Connect the popular Contact Form 7 to Laposta. Add a Laposta list and the Contact Form 7 code will be automatically generated for you. Add additional fields to complete your form.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-cf7" target="_blank" class="button button-primary">
            <?php esc_html_e('View Plugin', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

        <!-- Laposta for WPForms Plugin -->
        <div class="laposta-plugin-card">
          <h3><?php esc_html_e('Laposta for WPForms', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Connect the popular WPForms to Laposta.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-for-wpforms" target="_blank" class="button button-primary">
            <?php esc_html_e('View Plugin', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

        <!-- Laposta for WooCommerce Plugin -->
        <div class="laposta-plugin-card">
          <h3><?php esc_html_e('Laposta for WooCommerce', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Let your customers subscribe to your newsletter during checkout. Synchronize WooCommerce customer and order data with your Laposta fields. Start an automation or segment your customers based on their purchase behavior.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-for-woocommerce" target="_blank" class="button button-primary">
            <?php esc_html_e('View Plugin', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

        <!-- Laposta Auto Campaign Plugin -->
        <div class="laposta-plugin-card">
          <h3><?php esc_html_e('Laposta Auto Campaign', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Automatically send the latest matching campaign to new newsletter subscribers via Laposta webhooks.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-auto-campaign" target="_blank" class="button button-primary">
            <?php esc_html_e('View Plugin', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

        <!-- Laposta Campaigns (full version) -->
        <div class="laposta-plugin-card laposta-plugin-card-featured">
          <h3><?php esc_html_e('Laposta Campaigns (full version)', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Get the full version for unlimited campaigns, advanced filtering, custom layouts, and more features.', 'campaigns-lite-for-laposta'); ?></p>
          <a href="https://miix.dev/wp/laposta-campaigns" target="_blank" class="button button-primary">
            <?php esc_html_e('Get Full Version', 'campaigns-lite-for-laposta'); ?>
          </a>
        </div>

      </div>

      <p class="laposta-more-plugins-contact">
        <strong><?php esc_html_e('Need custom WordPress development?', 'campaigns-lite-for-laposta'); ?></strong><br>
        <a href="https://miix.dev/contact" target="_blank"><?php esc_html_e('Contact Miix for custom solutions', 'campaigns-lite-for-laposta'); ?></a>
      </p>
    </div>
    <?php
  }

  /**
   * Display shortcode settings
   */
  private function display_shortcode_settings()
  {
    ?>
    <div class="laposta-shortcode-settings">
      <h2><?php esc_html_e('Shortcode Settings', 'campaigns-lite-for-laposta'); ?></h2>

      <table class="form-table">
        <tr>
          <th scope="row"><?php esc_html_e('Basic Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>number</code> -
            <?php esc_html_e('Number of campaigns to display (default: 3, max: 6 in Lite version)', 'campaigns-lite-for-laposta'); ?><br>
            <code>show_screenshot</code> -
            <?php esc_html_e('Show campaign screenshots (yes/no)', 'campaigns-lite-for-laposta'); ?><br>
            <code>show_date</code> -
            <?php esc_html_e('Show campaign dates (yes/no)', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('Example Usage', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>[campaigns_lite_for_laposta]</code> -
            <?php esc_html_e('Shows 3 campaigns (default)', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta number="12"]</code> -
            <?php esc_html_e('Shows 12 campaigns (maximum)', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta show_screenshot="no"]</code> -
            <?php esc_html_e('Shows campaigns without screenshots', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
      </table>

      <h3 style="margin-top: 30px;"><?php esc_html_e('Full Version Features', 'campaigns-lite-for-laposta'); ?></h3>
      <p><?php esc_html_e('The following parameters are available in the full version only:', 'campaigns-lite-for-laposta'); ?></p>

      <table class="form-table">
        <tr>
          <th scope="row"><?php esc_html_e('Date Filtering Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>max_age</code> -
            <?php esc_html_e('Show campaigns not older than specified age (e.g., "30", "3m", "1y")', 'campaigns-lite-for-laposta'); ?><br>
            <code>current_year</code> -
            <?php esc_html_e('Show only campaigns from current year (yes/no)', 'campaigns-lite-for-laposta'); ?><br>
            <p class="description">
              <?php esc_html_e('Supported age formats: number (days), "3m" (months), "1y" (years)', 'campaigns-lite-for-laposta'); ?>
            </p>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('Layout Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>layout</code> -
            <?php esc_html_e('Display layout: "grid" (default) or "list"', 'campaigns-lite-for-laposta'); ?><br>
            <code>columns</code> -
            <?php esc_html_e('Number of columns for grid layout (1-6, default: 3)', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('Styling Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>container_class</code> -
            <?php esc_html_e('Custom CSS classes for the container (space-separated)', 'campaigns-lite-for-laposta'); ?><br>
            <code>item_class</code> -
            <?php esc_html_e('Custom CSS class for individual campaign items', 'campaigns-lite-for-laposta'); ?><br>
            <code>custom_css_only</code> -
            <?php esc_html_e('Use only custom classes, remove default plugin classes (yes/no)', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('Filtering Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>include</code> -
            <?php esc_html_e('Show only campaigns with these keywords in the subject (comma-separated)', 'campaigns-lite-for-laposta'); ?><br>
            <code>exclude</code> -
            <?php esc_html_e('Hide campaigns with these keywords in the subject (comma-separated)', 'campaigns-lite-for-laposta'); ?><br>
            <code>include_name</code> -
            <?php esc_html_e('Show only campaigns with these keywords in the campaign name (comma-separated)', 'campaigns-lite-for-laposta'); ?><br>
            <code>exclude_name</code> -
            <?php esc_html_e('Hide campaigns with these keywords in the campaign name (comma-separated)', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('List Format Parameters', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <code>list</code> -
            <?php esc_html_e('Simple list format: "ul" (bulleted list), "ol" (numbered list), or leave empty for divs', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
        <tr>
          <th scope="row"><?php esc_html_e('Example Usage', 'campaigns-lite-for-laposta'); ?></th>
          <td>
            <strong><?php esc_html_e('Basic Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta]</code> -
            <?php esc_html_e('Shows 5 campaigns with default settings', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta number="all"]</code> -
            <?php esc_html_e('Shows all campaigns', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta date_format="F j, Y"]</code> -
            <?php esc_html_e('Shows campaigns with custom date format (January 15, 2024)', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta date_format="d/m/Y"]</code> -
            <?php esc_html_e('Shows campaigns with European date format (15/01/2024)', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Date Filtering Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta max_age="30"]</code> -
            <?php esc_html_e('Shows campaigns from the last 30 days', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta max_age="3m"]</code> -
            <?php esc_html_e('Shows campaigns from the last 3 months', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta current_year="yes"]</code> -
            <?php esc_html_e('Shows only campaigns from current year', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta max_age="1y" number="all"]</code> -
            <?php esc_html_e('Shows all campaigns from the last year', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Layout Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta layout="grid" columns="4"]</code> -
            <?php esc_html_e('Shows campaigns in a 4-column grid with thumbnails', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta layout="list"]</code> -
            <?php esc_html_e('Shows campaigns in a minimal list format without thumbnails', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta layout="grid" columns="1"]</code> -
            <?php esc_html_e('Shows campaigns in a single column with thumbnails', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('List Format Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta list="ul"]</code> -
            <?php esc_html_e('Shows campaigns as a bulleted list', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta list="ol"]</code> -
            <?php esc_html_e('Shows campaigns as a numbered list', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta list="ul" show_date="yes"]</code> -
            <?php esc_html_e('Shows campaigns as a bulleted list with dates', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Styling Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta container_class="my-campaigns featured"]</code> -
            <?php esc_html_e('Adds custom CSS classes to the container', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta item_class="campaign-card"]</code> -
            <?php esc_html_e('Adds custom CSS class to each campaign item', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta container_class="highlighted" item_class="premium-card" layout="grid" columns="2"]</code> -
            <?php esc_html_e('Combines custom styling with layout options', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta container_class="my-custom-grid" custom_css_only="yes"]</code> -
            <?php esc_html_e('Uses only custom classes, removes all plugin CSS classes', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Filtering by Subject Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta exclude="internal,test"]</code> -
            <?php esc_html_e('Hides campaigns containing "internal" or "test" in the subject', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta include="sale"]</code> -
            <?php esc_html_e('Shows only campaigns containing "sale" in the subject', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Filtering by Name Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta exclude_name="internal,test"]</code> -
            <?php esc_html_e('Hides campaigns containing "internal" or "test" in the campaign name', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta include_name="newsletter"]</code> -
            <?php esc_html_e('Shows only campaigns containing "newsletter" in the campaign name', 'campaigns-lite-for-laposta'); ?><br>
            <br>
            <strong><?php esc_html_e('Combined Examples:', 'campaigns-lite-for-laposta'); ?></strong><br>
            <code>[campaigns_lite_for_laposta include="news" include_name="monthly" number="3" show_date="yes"]</code> -
            <?php esc_html_e('Shows 3 most recent campaigns with "news" in subject and "monthly" in name, with dates', 'campaigns-lite-for-laposta'); ?><br>
            <code>[campaigns_lite_for_laposta exclude="internal" exclude_name="test" number="3" show_screenshot="no"]</code> -
            <?php esc_html_e('Shows up to 3 campaigns except those with "internal" in subject or "test" in name, without screenshots', 'campaigns-lite-for-laposta'); ?>
          </td>
        </tr>
      </table>
    </div>
    <?php
  }

  /**
   * Setup help tabs
   */
  public function setup_help_tabs()
  {
    $screen = get_current_screen();

    // Only proceed if we have a screen
    if (!$screen) {
      return;
    }

    // Add instructions help tab
    $screen->add_help_tab(array(
      'id' => 'laposta-campaigns-instructions',
      'title' => __('Instructions', 'campaigns-lite-for-laposta'),
      'content' => $this->get_instructions_content()
    ));

    // Add shortcode help tab
    $screen->add_help_tab(array(
      'id' => 'laposta-campaigns-shortcode',
      'title' => __('Shortcode', 'campaigns-lite-for-laposta'),
      'content' => $this->get_shortcode_help_content()
    ));

    // Add account info help tab if available
    $account_info = $this->fetch_account_info();
    if (!is_wp_error($account_info) && !empty($account_info)) {
      $screen->add_help_tab(array(
        'id' => 'laposta-campaigns-account',
        'title' => __('Account Info', 'campaigns-lite-for-laposta'),
        'content' => $this->get_account_info_help_content($account_info)
      ));
    }

    // Set help sidebar
    $screen->set_help_sidebar(
      '<p><strong>' . esc_html__('For more information:', 'campaigns-lite-for-laposta') . '</strong></p>' .
      '<p>' . esc_html__('Questions about the plugin:', 'campaigns-lite-for-laposta') . '</p>' .
      '<p><a href="https://miix.dev/contact" target="_blank">' . esc_html__('Contact Miix', 'campaigns-lite-for-laposta') . '</a></p>' .
      '<p>' . esc_html__('Questions about your Laposta account:', 'campaigns-lite-for-laposta') . '</p>' .
      '<p><a href="https://laposta.nl/helpdesk" target="_blank">' . esc_html__('Contact Laposta Helpdesk', 'campaigns-lite-for-laposta') . '</a></p>'
    );
  }

  /**
   * Get formatted instructions content for help tab
   *
   * @return string Formatted HTML content
   */
  private function get_instructions_content()
  {
    $content = '<h2>' . esc_html__('Instructions', 'campaigns-lite-for-laposta') . '</h2>';
    $content .= '<ol>';
    $content .= '<li>' . esc_html__('Enter your Laposta API Key and click "Save Settings".', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '<li>' . esc_html__('Click the Campaigns tab to view your Laposta campaigns (maximum 12 in Lite version).', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '<li>' . esc_html__('Use the shortcode [campaigns_lite_for_laposta] to display campaigns on your pages (see Shortcode tab for details).', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '</ol>';

    $content .= '<p style="background: #fff; border-left: 4px solid #0073aa; padding: 10px; margin: 15px 0;">';
    $content .= '<strong>' . esc_html__('Lite Version:', 'campaigns-lite-for-laposta') . '</strong> ';
    $content .= sprintf(
      /* translators: %s: link to the full version plugin */
      esc_html__('This lite version is limited to 12 campaigns with basic display options. Get the %s for unlimited campaigns and advanced features.', 'campaigns-lite-for-laposta'),
      '<a href="https://miix.dev/wp/laposta-campaigns" target="_blank">' . esc_html__('full version', 'campaigns-lite-for-laposta') . '</a>'
    );
    $content .= '</p>';

    return $content;
  }

  /**
   * Get formatted shortcode help content
   *
   * @return string Formatted HTML content
   */
  private function get_shortcode_help_content()
  {
    $content = '<h2>' . esc_html__('Shortcode Usage - Lite Version', 'campaigns-lite-for-laposta') . '</h2>';
    $content .= '<p>' . esc_html__('Use the shortcode [campaigns_lite_for_laposta] to display your campaigns in posts or pages.', 'campaigns-lite-for-laposta') . '</p>';

    $content .= '<h3>' . esc_html__('Available Parameters', 'campaigns-lite-for-laposta') . '</h3>';
    $content .= '<ul>';
    $content .= '<li><code>number</code> - ' . esc_html__('Number of campaigns to display (default: 3, maximum: 6)', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '<li><code>show_screenshot</code> - ' . esc_html__('Show campaign screenshots (default: yes)', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '<li><code>show_date</code> - ' . esc_html__('Show campaign dates (default: yes)', 'campaigns-lite-for-laposta') . '</li>';
    $content .= '</ul>';

    $content .= '<h3>' . esc_html__('Examples', 'campaigns-lite-for-laposta') . '</h3>';
    $content .= '<p><code>[campaigns_lite_for_laposta]</code> - ' . esc_html__('Shows 3 campaigns (default)', 'campaigns-lite-for-laposta') . '</p>';
    $content .= '<p><code>[campaigns_lite_for_laposta number="12"]</code> - ' . esc_html__('Shows 12 campaigns (maximum)', 'campaigns-lite-for-laposta') . '</p>';
    $content .= '<p><code>[campaigns_lite_for_laposta show_screenshot="no"]</code> - ' . esc_html__('Shows campaigns without screenshots', 'campaigns-lite-for-laposta') . '</p>';

    $content .= '<div style="background: #fff; border-left: 4px solid #0073aa; padding: 10px; margin: 15px 0;">';
    $content .= '<p style="margin: 0;"><strong>' . esc_html__('Want more features?', 'campaigns-lite-for-laposta') . '</strong></p>';
    $content .= '<p style="margin: 5px 0 0 0;">' . sprintf(
      /* translators: %s: link to the full version plugin */
      esc_html__('Get the %s for unlimited campaigns, advanced filtering, custom layouts, date filtering, and more!', 'campaigns-lite-for-laposta'),
      '<a href="https://miix.dev/wp/laposta-campaigns" target="_blank">' . esc_html__('full version', 'campaigns-lite-for-laposta') . '</a>'
    ) . '</p>';
    $content .= '</div>';

    return $content;
  }

  /**
   * Get formatted account help content
   *
   * @param array $account_info Account information
   * @return string Formatted HTML content
   */
  private function get_account_info_help_content($account_info)
  {
    $content = '<h2>' . esc_html__('Information about your Laposta account', 'campaigns-lite-for-laposta') . '</h2>';

    if (!empty($account_info) && isset($account_info['name']) && isset($account_info['number'])) {
      $content .= '<p><strong>' . esc_html__('Account Name:', 'campaigns-lite-for-laposta') . '</strong> ' . esc_html($account_info['name']) . '</p>';

      // Account number with copy button - using the same ID format as the main page for JS compatibility
      $content .= '<p class="laposta-account-number">';
      $content .= '<strong>' . esc_html__('Account Number:', 'campaigns-lite-for-laposta') . '</strong> ';
      $content .= '<span>' . esc_html($account_info['number']) . '</span> ';
      $content .= '<button id="copy-account-number" class="button" data-account-number="' . esc_attr($account_info['number']) . '">';
      $content .= esc_html__('Copy to Clipboard', 'campaigns-lite-for-laposta');
      $content .= '</button>';
      $content .= '</p>';

      $content .= '<p>' . esc_html__('If you need assistance, please provide this information to Laposta helpdesk.', 'campaigns-lite-for-laposta') . '</p>';
    } else {
      $content .= '<p>' . esc_html__('No account information available. Please ensure your API key is correctly set.', 'campaigns-lite-for-laposta') . '</p>';
    }

    return $content;
  }

  /**
   * Display help tab notice
   */
  private function display_instructions()
  {
    $account_info = $this->fetch_account_info();

    ?>
    <div style="display: flex; flex-direction: column;">
    <?php
    
    // Show notice about help tab
    ?>
    <div class="notice notice-info" style="order: 1;">
      <p>
        <?php
        /* translators: This is the notice directing users to the help tab */
        esc_html_e('Need help? Click the Help tab in the upper-right corner of the screen for instructions and account information.', 'campaigns-lite-for-laposta');
        ?>
      </p>
    </div>
    <?php


    // Show error notices if API key issue
    if (is_wp_error($account_info)) {
      $error_code = $account_info->get_error_code();

      if ($error_code === 'no_api_key') {
        ?>
        <div class="notice notice-warning" style="order: 0;">
          <p><strong><?php esc_html_e('Please select your Laposta API Key in the field below.', 'campaigns-lite-for-laposta'); ?></strong>
          </p>
        </div>
        <?php
      } elseif ($error_code !== 'too_many_requests') {
        ?>
        <div class="notice notice-error" style="order: 0;">
          <p>
            <strong><?php esc_html_e('Unable to retrieve account information. Please ensure your API key is correct.', 'campaigns-lite-for-laposta'); ?></strong>
          </p>
        </div>
        <?php
      }
    }
    
    ?>
    </div>
    <?php
  }

  /**
   * Display campaigns
   */
  private function display_campaigns()
  {
    // Check if API key is selected
    $api_key = self::get_actual_api_key();
    if (empty($api_key)) {
      echo '<div class="notice notice-warning"><p>';
      printf(
        // Translators: %1$s is the opening HTML link tag, %2$s is the closing HTML link tag
        esc_html__('Please select an API key in the %1$sSettings tab%2$s to view campaigns.', 'campaigns-lite-for-laposta'),
        '<a href="' . esc_url(admin_url('admin.php?page=campaigns-lite-for-laposta-settings&tab=settings')) . '">',
        '</a>'
      );
      echo '</p></div>';
      return;
    }

    // LITE VERSION: Display limitation notice
    echo '<div class="notice notice-info"><p>';
    echo '<strong>' . esc_html__('Campaigns Lite for Laposta', 'campaigns-lite-for-laposta') . ':</strong> ';
    echo sprintf(
      /* translators: %s: link to the full version plugin */
      esc_html__('This lite version displays a maximum of 12 campaigns. Get the %s for unlimited campaigns, advanced filtering, custom layouts, and more features.', 'campaigns-lite-for-laposta'),
      '<a href="https://miix.dev/wp/laposta-campaigns" target="_blank">' . esc_html__('full version', 'campaigns-lite-for-laposta') . '</a>'
    );
    echo '</p></div>';

    // Check if an API key is configured
    $api_key = self::get_actual_api_key();
    if (empty($api_key)) {
      echo '<div class="notice notice-warning"><p>';
      printf(
        // Translators: %1$s is the opening HTML link tag, %2$s is the closing HTML link tag
        esc_html__('No API key configured. Please add an API key in the %1$sSettings tab%2$s.', 'campaigns-lite-for-laposta'),
        '<a href="' . esc_url(admin_url('admin.php?page=campaigns-lite-for-laposta-settings&tab=settings')) . '">',
        '</a>'
      );
      echo '</p></div>';
      return;
    }

    // Check if we can use cached campaigns - use a key that includes the API key
    $all_campaigns_cache_key = 'laposta_all_campaigns_' . md5($api_key);
    $cached_all_campaigns = get_transient($all_campaigns_cache_key);

    if ($cached_all_campaigns !== false) {
      $response = $cached_all_campaigns;
    } else {
      // Try the regular API method first
      $response = $this->api->get_campaigns();

      // If the response is an error or empty, try a direct API call as a fallback
      if (is_wp_error($response) || empty($response)) {
        // Get the API key
        $api_key = self::get_actual_api_key();

        if (!empty($api_key)) {
          // Make a direct API request to Laposta
          $endpoint = defined('LAPOSTA_API_ENDPOINT') ? LAPOSTA_API_ENDPOINT : 'https://api.laposta.org/v2/';
          $direct_url = $endpoint . 'campaign';
          $direct_args = array(
            'headers' => array(
              'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
              'Content-Type' => 'application/json',
            ),
            'timeout' => 15,
          );

          $direct_response = wp_remote_get($direct_url, $direct_args);

          if (!is_wp_error($direct_response)) {
            $direct_body = wp_remote_retrieve_body($direct_response);
            $direct_data = json_decode($direct_body, true);

            if (isset($direct_data['data']) && is_array($direct_data['data'])) {
              // Override the response with our direct call result
              $response = $direct_data['data'];
            }
          }
        }
      }

      // Cache valid responses for 4 hours
      if (!is_wp_error($response) && !empty($response)) {
        set_transient($all_campaigns_cache_key, $response, 4 * HOUR_IN_SECONDS);
        $this->track_transient($all_campaigns_cache_key);
      }
    }

    if (is_wp_error($response)) {
      $error_code = $response->get_error_code();
      $error_message = $response->get_error_message();

      // Customize error message for common errors
      if ($error_code === 'no_api_instance') {
        echo '<div class="notice notice-error"><p>' .
          sprintf(
            // Translators: %1$s is the opening HTML link tag, %2$s is the closing HTML link tag
            esc_html__('API connection not available. Please make sure you have selected a valid API key in the %1$sSettings tab%2$s.', 'campaigns-lite-for-laposta'),
            '<a href="' . esc_url(admin_url('admin.php?page=campaigns-lite-for-laposta-settings&tab=settings')) . '">',
            '</a>'
          ) .
          '</p>';

        echo '<p>' . esc_html__('Please make sure that:', 'campaigns-lite-for-laposta') . '</p>';
        echo '<ol>';
        echo '<li>' . esc_html__('You have entered a valid Laposta API key in the Settings tab', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('Your API key has the necessary permissions to access campaigns', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('Your server can connect to api.laposta.org', 'campaigns-lite-for-laposta') . '</li>';
        echo '</ol>';
        echo '</div>';
      } else if ($error_code === 'api_error' || $error_code === 'json_error') {
        echo '<div class="notice notice-error"><p>' . esc_html($error_message) . '</p>';
        echo '<p>' . esc_html__('There was an error communicating with the Laposta API. Please check:', 'campaigns-lite-for-laposta') . '</p>';
        echo '<ol>';
        echo '<li>' . esc_html__('Your API key is valid and has the correct permissions', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('Your server can connect to api.laposta.org', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('The WordPress error logs for more details', 'campaigns-lite-for-laposta') . '</li>';
        echo '</ol>';
        echo '</div>';
      } else {
        echo '<div class="notice notice-error"><p>' . esc_html($error_message) . '</p></div>';
      }

      // Show technical details for troubleshooting
      ?>
      <div class="notice notice-info">
        <p><strong><?php esc_html_e('Technical Details:', 'campaigns-lite-for-laposta'); ?></strong></p>
        <p>
          <strong><?php esc_html_e('Error Code:', 'campaigns-lite-for-laposta'); ?></strong> <?php echo esc_html($error_code); ?><br>
          <strong><?php esc_html_e('Error Message:', 'campaigns-lite-for-laposta'); ?></strong>
          <?php echo esc_html($response->get_error_message()); ?><br>
          <strong><?php esc_html_e('Selected API Key:', 'campaigns-lite-for-laposta'); ?></strong>
          <?php echo esc_html(get_option('campaigns_lite_for_laposta_api_key') ? 'Yes (Key starts with "' . esc_html(substr(get_option('campaigns_lite_for_laposta_api_key'), 0, 5)) . '...")' : 'No key selected'); ?>
        </p>
      </div>
      <?php
      return;
    }

    if (empty($response)) {
      echo '<div class="notice notice-info">';
      echo '<p>' . esc_html__('No campaigns found. Your API connection is working, but no campaigns were returned.', 'campaigns-lite-for-laposta') . '</p>';
      echo '<p>' . esc_html__('This could be because:', 'campaigns-lite-for-laposta') . '</p>';
      echo '<ul>';
      echo '<li>' . esc_html__('You have not created any campaigns in your Laposta account yet', 'campaigns-lite-for-laposta') . '</li>';
      echo '<li>' . esc_html__('The API key you are using does not have access to any campaigns', 'campaigns-lite-for-laposta') . '</li>';
      echo '</ul>';

      // API Testing Tool - Allow admin to see what's happening
      if (current_user_can('manage_options')) {
        $api_key = self::get_actual_api_key();

        echo '<div style="margin-top: 20px; padding: 10px; background: #f8f9fa; border: 1px solid #ddd; border-radius: 4px;">';
        echo '<h3>' . esc_html__('Laposta API Test (Admin Only)', 'campaigns-lite-for-laposta') . '</h3>';
        echo '<p>' . esc_html__('API Key:', 'campaigns-lite-for-laposta') . ' ' . (!empty($api_key) ? 'Set (starts with: ' . esc_html(substr($api_key, 0, 5)) . '...)' : 'Not set') . '</p>';

        // Make a test request to check API key validity
        $endpoint = defined('LAPOSTA_API_ENDPOINT') ? LAPOSTA_API_ENDPOINT : 'https://api.laposta.org/v2/';
        $test_url = $endpoint . 'auth';
        $test_args = array(
          'headers' => array(
            'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
          ),
          'timeout' => 15,
        );

        $test_response = wp_remote_get($test_url, $test_args);

        if (!is_wp_error($test_response)) {
          $test_code = wp_remote_retrieve_response_code($test_response);
          $test_body = wp_remote_retrieve_body($test_response);
          $test_data = json_decode($test_body, true);

          echo '<p>' . esc_html__('Authentication Test:', 'campaigns-lite-for-laposta') . ' ';

          if ($test_code === 200 && isset($test_data['authenticated']) && $test_data['authenticated'] === true) {
            echo '<span style="color: green; font-weight: bold;">' . esc_html__('Successful', 'campaigns-lite-for-laposta') . '</span>';
            if (isset($test_data['account_name'])) {
              echo ' - ' . esc_html__('Account:', 'campaigns-lite-for-laposta') . ' ' . esc_html($test_data['account_name']);
            }
          } else {
            echo '<span style="color: red; font-weight: bold;">' . esc_html__('Failed', 'campaigns-lite-for-laposta') . '</span>';
            echo ' - ' . esc_html__('Status Code:', 'campaigns-lite-for-laposta') . ' ' . esc_html($test_code);
          }

          echo '</p>';

          // Try fetching campaigns directly
          $endpoint = defined('LAPOSTA_API_ENDPOINT') ? LAPOSTA_API_ENDPOINT : 'https://api.laposta.org/v2/';
          $campaigns_url = $endpoint . 'campaign';
          $campaigns_args = array(
            'headers' => array(
              'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
              'Content-Type' => 'application/json',
            ),
            'timeout' => 15,
          );

          $campaigns_response = wp_remote_get($campaigns_url, $campaigns_args);

          if (!is_wp_error($campaigns_response)) {
            $campaigns_code = wp_remote_retrieve_response_code($campaigns_response);
            $campaigns_body = wp_remote_retrieve_body($campaigns_response);
            $campaigns_data = json_decode($campaigns_body, true);

            echo '<p>' . esc_html__('Campaign API Test:', 'campaigns-lite-for-laposta') . ' ';

            if ($campaigns_code === 200) {
              echo '<span style="color: green; font-weight: bold;">' . esc_html__('Successful', 'campaigns-lite-for-laposta') . '</span>';
              if (isset($campaigns_data['data']) && is_array($campaigns_data['data'])) {
                echo ' - ' . esc_html__('Found:', 'campaigns-lite-for-laposta') . ' ' . count($campaigns_data['data']) . ' ' . esc_html__('campaigns', 'campaigns-lite-for-laposta');
              } else {
                echo ' - ' . esc_html__('No campaigns in the response', 'campaigns-lite-for-laposta');
              }
            } else {
              echo '<span style="color: red; font-weight: bold;">' . esc_html__('Failed', 'campaigns-lite-for-laposta') . '</span>';
              echo ' - ' . esc_html__('Status Code:', 'campaigns-lite-for-laposta') . ' ' . esc_html($campaigns_code);
            }

            echo '</p>';

            if (isset($campaigns_data['data']) && is_array($campaigns_data['data']) && !empty($campaigns_data['data'])) {
              echo '<p>' . esc_html__('There are campaigns available through the API but they are not displaying. This is likely a data format issue.', 'campaigns-lite-for-laposta') . '</p>';

              // Campaign data is available but not displaying properly
              echo '<p>' . esc_html__('Campaign data is available but may have formatting issues.', 'campaigns-lite-for-laposta') . '</p>';
            }
          } else {
            echo '<p>' . esc_html__('Campaign API Test:', 'campaigns-lite-for-laposta') . ' <span style="color: red; font-weight: bold;">' . esc_html__('Failed with error', 'campaigns-lite-for-laposta') . '</span> - ' . esc_html($campaigns_response->get_error_message()) . '</p>';
          }
        } else {
          echo '<p>' . esc_html__('Authentication Test:', 'campaigns-lite-for-laposta') . ' <span style="color: red; font-weight: bold;">' . esc_html__('Failed with error', 'campaigns-lite-for-laposta') . '</span> - ' . esc_html($test_response->get_error_message()) . '</p>';
        }

        echo '</div>';
      }

      echo '</div>';
      return;
    }

    // Separate sent and concept campaigns
    $sent_campaigns = array();
    $concept_campaigns = array();

    // The response should already be normalized by the API class get_campaigns() method
    // Process the campaigns
    foreach ($response as $item) {
      // Skip invalid items
      if (!isset($item['campaign'])) {
        continue;
      }

      // Skip items without campaign ID
      if (empty($item['campaign']['campaign_id'])) {
        continue;
      }

      // Sort into sent or concept based on delivery date
      if (!empty($item['campaign']['delivery_ended'])) {
        $sent_campaigns[] = $item;
      } else {
        $concept_campaigns[] = $item;
      }
    }

    // If we have no campaigns after processing, show info for admins
    if (empty($sent_campaigns) && empty($concept_campaigns) && current_user_can('manage_options')) {
      echo '<div class="notice notice-warning">';
      echo '<p>' . esc_html__('No campaigns could be processed from the API response.', 'campaigns-lite-for-laposta') . '</p>';
      echo '</div>';
    }

    // Note: get_campaigns() already sorts campaigns appropriately,
    // but we'll keep this sorting in case we got here through a different path

    // Sort sent campaigns by delivery date (newest first)
    usort($sent_campaigns, function ($a, $b) {
      $date_a = strtotime($a['campaign']['delivery_ended']);
      $date_b = strtotime($b['campaign']['delivery_ended']);
      return $date_b - $date_a;
    });

    // Sort concept campaigns by creation date (newest first)
    usort($concept_campaigns, function ($a, $b) {
      $date_a = strtotime($a['campaign']['created']);
      $date_b = strtotime($b['campaign']['created']);
      return $date_b - $date_a;
    });

    // Merge sorted campaigns - sent campaigns first, then concept campaigns
    $sorted_campaigns = array_merge($sent_campaigns, $concept_campaigns);

    // LITE VERSION: Limit to maximum 12 campaigns
    $sorted_campaigns = array_slice($sorted_campaigns, 0, 12);

    // Since we now get complete campaign data from the direct API call including screenshots and web URLs,
    // there's no need to enrich individual campaigns anymore. We just cache the sorted list.
    $cache_key = 'campaigns_lite_for_laposta_list_' . md5($api_key);
    $cached_campaigns = get_transient($cache_key);

    // Use cached campaigns if available
    if ($cached_campaigns !== false) {
      $sorted_campaigns = $cached_campaigns;
    } else {
      // Cache the sorted campaigns list for 2 hours to improve performance
      set_transient($cache_key, $sorted_campaigns, 2 * HOUR_IN_SECONDS);
      $this->track_transient($cache_key);
    }

    // Display campaigns (limited to 12, no pagination UI)
    $campaigns = $sorted_campaigns;

    ?>
    <div class="laposta-campaigns-list">
      <table class="wp-list-table widefat fixed striped">
        <thead>
          <tr>
            <th><?php esc_html_e('Campaign Name', 'campaigns-lite-for-laposta'); ?></th>
            <th><?php esc_html_e('Subject', 'campaigns-lite-for-laposta'); ?></th>
            <th><?php esc_html_e('Status', 'campaigns-lite-for-laposta'); ?></th>
            <th><?php esc_html_e('Sent date', 'campaigns-lite-for-laposta'); ?></th>
            <th><?php esc_html_e('Screenshot', 'campaigns-lite-for-laposta'); ?></th>
            <th><?php esc_html_e('Actions', 'campaigns-lite-for-laposta'); ?></th>
          </tr>
        </thead>
        <tbody>
          <?php
          foreach ($campaigns as $item):
            if (!isset($item['campaign'])) {
              continue;
            }

            $campaign = $item['campaign'];
            if (empty($campaign['campaign_id'])) {
              continue;
            }

            // Process campaign data
      
            // Determine the name field - it could be 'name', 'title', or custom fallback
            $campaign_name = '';
            if (!empty($campaign['name'])) {
              $campaign_name = $campaign['name'];
            } else if (!empty($campaign['title'])) {
              $campaign_name = $campaign['title'];
            } else {
              // Fallback to subject or campaign ID if name is not available
              $campaign_name = !empty($campaign['subject']) ? $campaign['subject'] : 'Campaign ' . $campaign['campaign_id'];
            }
            ?>
            <tr>
              <td>
                <strong><?php echo esc_html($campaign_name); ?></strong>
              </td>
              <td><?php echo esc_html($campaign['subject'] ?? ''); ?></td>
              <td>
                <span class="status-badge status-<?php echo !empty($campaign['delivery_ended']) ? 'sent' : 'concept'; ?>">
                  <?php echo !empty($campaign['delivery_ended']) ?
                    esc_html__('Sent', 'campaigns-lite-for-laposta') :
                    esc_html__('Concept', 'campaigns-lite-for-laposta');
                  ?>
                </span>
              </td>
              <td>
                <?php
                if (isset($campaign['delivery_ended'])) {
                  $date = strtotime($campaign['delivery_ended']);
                  echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $date));
                }
                ?>
              </td>
              <td>
                <?php
                if (!empty($campaign['delivery_ended'])) { // Only show for sent campaigns
                  $screenshot_url = '';

                  if (!empty($campaign['screenshot'])) {
                    // Try to get the best size for display
                    if (isset($campaign['screenshot']['226x268'])) {
                      $screenshot_url = $campaign['screenshot']['226x268'];
                    } elseif (is_array($campaign['screenshot'])) {
                      // Try some alternate sizes that might be available
                      $possible_sizes = ['200x236', '300x354', '400x472', '320x378'];
                      foreach ($possible_sizes as $size) {
                        if (isset($campaign['screenshot'][$size])) {
                          $screenshot_url = $campaign['screenshot'][$size];
                          break;
                        }
                      }

                      // If no specific size found, just grab the first available one
                      if (empty($screenshot_url)) {
                        $screenshot_url = reset($campaign['screenshot']);
                      }
                    }
                  }

                  if (!empty($screenshot_url)) {
                    ?>
                    <img src="<?php echo esc_url($screenshot_url); ?>" alt="<?php echo esc_attr($campaign_name); ?>"
                      style="width: 113px; height: 134px; object-fit: cover;" />
                    <?php
                  } elseif (current_user_can('manage_options')) {
                    ?>
                    <div style="font-size: 10px; color: #999;">
                      <?php
                      if (empty($campaign['screenshot'])) {
                        esc_html_e('No screenshot data', 'campaigns-lite-for-laposta');
                      } elseif (is_array($campaign['screenshot'])) {
                        esc_html_e('Available sizes: ', 'campaigns-lite-for-laposta');
                        echo esc_html(implode(', ', array_keys($campaign['screenshot'])));
                      } else {
                        esc_html_e('Screenshot in unexpected format', 'campaigns-lite-for-laposta');
                      }
                      ?>
                    </div>
                    <?php
                  }
                }
                ?>
              </td>
              <td class="column-actions">
                <?php if (!empty($campaign['delivery_ended']) && !empty($campaign['web'])): ?>
                  <a href="<?php echo esc_url($campaign['web']); ?>" target="_blank" class="button">
                    <?php esc_html_e('View Online', 'campaigns-lite-for-laposta'); ?>
                  </a>
                <?php elseif (!empty($campaign['delivery_ended']) && current_user_can('manage_options')): ?>
                  <div style="font-size: 10px; color: #999;">
                    <?php esc_html_e('No web URL available', 'campaigns-lite-for-laposta'); ?>
                  </div>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>

      <?php if (current_user_can('manage_options')): ?>
        <div class="admin-tools"
          style="margin: 15px 0; padding: 10px; background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px;">
          <h3><?php esc_html_e('Admin Tools', 'campaigns-lite-for-laposta'); ?></h3>
          <p><?php esc_html_e('Use this tool to refresh the campaign data.', 'campaigns-lite-for-laposta'); ?></p>

          <div class="admin-tools-buttons" style="display: flex; gap: 10px;">
            <button id="clear-campaign-cache" class="button button-primary">
              <?php esc_html_e('Clear Cache', 'campaigns-lite-for-laposta'); ?>
            </button>
            <div class="button-description" style="font-size: 12px; color: #666; align-self: center; margin-left: 10px;">
              <?php esc_html_e('This will clear cached data and fetch fresh campaign information on the next page load.', 'campaigns-lite-for-laposta'); ?>
            </div>
          </div>

          <div class="admin-tools-status" style="margin-top: 10px; font-size: 12px;">
            <p>
              <strong><?php esc_html_e('Cache Status:', 'campaigns-lite-for-laposta'); ?></strong>
              <?php
              if ($cached_campaigns !== false) {
                echo esc_html__('Using cached data (expires in ', 'campaigns-lite-for-laposta') .
                  esc_html(human_time_diff(time(), time() + get_option('_transient_timeout_' . $cache_key, 0))) .
                  ')';
              } else {
                echo esc_html__('Fresh data', 'campaigns-lite-for-laposta');
              }
              ?>
            </p>

            <!-- No additional description needed -->
          </div>
        </div>
      <?php endif; ?>
      <?php
      ?>
    </div>
    <?php
  }
}