<?php
/**
 * Laposta Campaigns API Class
 *
 * @package Campaigns_Lite_For_Laposta
 */

if (!defined('ABSPATH')) {
  exit;
}

/**
 * Direct API client utilities
 * 
 * Helper methods for making direct API calls to Laposta
 */
class Campaigns_Lite_For_Laposta_API_Utils 
{
  /**
   * API endpoint URL
   */
  /**
   * Get the API endpoint URL
   * Uses the centralized constant from laposta-api plugin if available
   * 
   * @return string The API endpoint URL
   */
  private static function get_api_endpoint()
  {
    return defined('LAPOSTA_API_ENDPOINT') ? LAPOSTA_API_ENDPOINT : 'https://api.laposta.org/v2/';
  }
  
  /**
   * Whether logging is enabled
   */
  private static $log_enabled = false;
  
  /**
   * Enable or disable logging
   * 
   * @param bool $enabled Whether to enable logging
   */
  public static function set_logging($enabled)
  {
    self::$log_enabled = $enabled;
  }
  
  /**
   * Make a GET request to the Laposta API
   *
   * @param string $api_key The API key
   * @param string $endpoint The API endpoint
   * @param array $params Query parameters
   * @return array|WP_Error The API response or error
   */
  public static function get($api_key, $endpoint, $params = array())
  {
    $url = self::get_api_endpoint() . $endpoint;
    
    if (!empty($params)) {
      $url = add_query_arg($params, $url);
    }
    
    $args = array(
      'headers' => array(
        'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        'Content-Type' => 'application/json',
      ),
      'timeout' => 30, // Increased timeout for potentially slow API responses
    );
    
    $response = wp_remote_get($url, $args);
    
    return self::handle_response($response);
  }
  
  /**
   * Make a POST request to the Laposta API
   *
   * @param string $api_key The API key
   * @param string $endpoint The API endpoint
   * @param array $data The data to send
   * @return array|WP_Error The API response or error
   */
  public static function post($api_key, $endpoint, $data = array())
  {
    $url = self::get_api_endpoint() . $endpoint;
    
    $args = array(
      'method' => 'POST',
      'headers' => array(
        'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        'Content-Type' => 'application/json',
      ),
      'body' => wp_json_encode($data),
      'timeout' => 30,
    );
    
    $response = wp_remote_post($url, $args);
    
    return self::handle_response($response);
  }
  
  /**
   * Handle an API response
   *
   * @param WP_Error|array $response The response from wp_remote_*
   * @return array|WP_Error The processed response data or error
   */
  private static function handle_response($response)
  {
    if (is_wp_error($response)) {
      return $response;
    }
    
    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    
    $parsed = json_decode($body, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new WP_Error('json_error', 'Failed to parse JSON response: ' . json_last_error_msg());
    }
    
    if ($code !== 200 && $code !== 201) {
      if (isset($parsed['error']) && !empty($parsed['error'])) {
        $error_message = isset($parsed['error']['message']) ? $parsed['error']['message'] : 'Unknown API error';
        return new WP_Error('api_error', $error_message);
      }
    }
    
    return $parsed;
  }
  
  /**
   * Get all campaigns from the API with caching
   *
   * @param string $api_key The API key
   * @param array $params Query parameters
   * @param int $cache_time How long to cache the response in seconds
   * @return array|WP_Error The campaigns data or error
   */
  public static function get_all_campaigns($api_key, $params = array(), $cache_time = 3600)
  {
    // Check if we can use a cached response
    $cache_key = 'campaigns_lite_campaigns_all_' . md5($api_key . serialize($params));
    $cached_response = get_transient($cache_key);
    
    if ($cached_response !== false) {
      return $cached_response;
    }
    
    $response = self::get($api_key, 'campaign', $params);
    
    if (is_wp_error($response)) {
      // Cache errors for a short time to avoid hammering the API
      set_transient($cache_key, $response, 5 * MINUTE_IN_SECONDS);
      return $response;
    }
    
    // Check if we got a valid response with data
    if (!isset($response['data']) || !is_array($response['data'])) {
      // Check if the response itself is the data (for API version differences)
      if (is_array($response) && isset($response[0]) && isset($response[0]['campaign'])) {
        // Cache the valid response
        set_transient($cache_key, $response, $cache_time);
        return $response;
      }
      
      $error = new WP_Error('invalid_response', __('Invalid response from Laposta API', 'campaigns-lite-for-laposta'));
      set_transient($cache_key, $error, 5 * MINUTE_IN_SECONDS);
      return $error;
    }
    
    // Cache the valid response
    set_transient($cache_key, $response['data'], $cache_time);
    return $response['data'];
  }
}

/**
 * API Handler class
 */
class Campaigns_Lite_For_Laposta_API
{
  /**
   * API Key
   *
   * @var string
   */
  private $api_key;
  
  /**
   * Whether logging is enabled
   *
   * @var bool
   */
  private $logging_enabled = false;

  /**
   * Constructor
   * 
   * @param bool $enable_logging Whether to enable verbose logging (default: false)
   */
  public function __construct($enable_logging = false)
  {
    // Get the actual API key using the settings class
    if (class_exists('Campaigns_Lite_For_Laposta_Settings')) {
      $this->api_key = Campaigns_Lite_For_Laposta_Settings::get_actual_api_key();
    } else {
      // Fallback to direct option access (shouldn't happen in normal operation)
      $this->api_key = get_option('campaigns_lite_for_laposta_api_key');
    }

    // Enable or disable logging
    $this->set_logging($enable_logging);
  }
  
  /**
   * Enable or disable logging for API operations
   * 
   * @param bool $enabled Whether to enable logging
   */
  public function set_logging($enabled)
  {
    $this->logging_enabled = $enabled;
    
    // Set logging on the API Utils class
    Campaigns_Lite_For_Laposta_API_Utils::set_logging($enabled);
  }

  /**
   * Get campaigns
   * 
   * @return array|WP_Error
   */
  public function get_campaigns()
  {
    // Check for a cached response
    $cache_key = 'campaigns_lite_all_campaigns_' . md5($this->api_key);
    $cached_response = get_transient($cache_key);
    
    if ($cached_response !== false) {
      return $cached_response;
    }
    
    // Verify API key is available
    if (empty($this->api_key)) {
      return new WP_Error('no_api_key', __('API key is not set. Please select a valid API key.', 'campaigns-lite-for-laposta'));
    }

    // Get campaigns through direct API call
    $result = Campaigns_Lite_For_Laposta_API_Utils::get_all_campaigns($this->api_key);
    
    if (is_wp_error($result)) {
      return $result;
    }
    
    // Make sure we have an array to work with
    if (!is_array($result)) {
      return array();
    }
    
    // Normalize the campaign data to ensure we have the expected structure
    $normalized_result = array();
    foreach ($result as $item) {
      // Check if this item already has a 'campaign' wrapper
      if (isset($item['campaign'])) {
        $normalized_result[] = $item;
      } 
      // Check if this is a direct campaign object
      else if (isset($item['campaign_id'])) {
        $normalized_result[] = array('campaign' => $item);
      }
    }
    
    // Sort campaigns by delivery date for sent campaigns, creation date for others (newest first)
    usort($normalized_result, function ($a, $b) {
      // Both are sent campaigns, sort by delivery date
      if (!empty($a['campaign']['delivery_ended']) && !empty($b['campaign']['delivery_ended'])) {
        return strtotime($b['campaign']['delivery_ended']) - strtotime($a['campaign']['delivery_ended']);
      }
      // A is sent, B is not, A comes first
      else if (!empty($a['campaign']['delivery_ended']) && empty($b['campaign']['delivery_ended'])) {
        return -1;
      }
      // B is sent, A is not, B comes first
      else if (empty($a['campaign']['delivery_ended']) && !empty($b['campaign']['delivery_ended'])) {
        return 1;
      }
      // Both are not sent, sort by creation date
      else {
        $date_a = isset($a['campaign']['created']) ? strtotime($a['campaign']['created']) : 0;
        $date_b = isset($b['campaign']['created']) ? strtotime($b['campaign']['created']) : 0;
        return $date_b - $date_a; // Descending order
      }
    });
    
    // Cache the result for 1 hour
    set_transient($cache_key, $normalized_result, HOUR_IN_SECONDS);
    
    return $normalized_result;
  }

  /**
   * Get campaign stats
   * 
   * @param string $campaign_id Campaign ID
   * @return array|WP_Error
   */
  public function get_campaign_stats($campaign_id)
  {
    if (empty($campaign_id)) {
      return new WP_Error('invalid_campaign_id', __('Invalid campaign ID', 'campaigns-lite-for-laposta'));
    }

    // Verify API key is available
    if (empty($this->api_key)) {
      return new WP_Error('no_api_key', __('API key is not set. Please select a valid API key.', 'campaigns-lite-for-laposta'));
    }

    // Check for a cached response
    $cache_key = 'laposta_campaign_stats_' . $campaign_id;
    $cached_response = get_transient($cache_key);
    
    if ($cached_response !== false) {
      return $cached_response;
    }
    
    // Make direct API call for stats
    $response = Campaigns_Lite_For_Laposta_API_Utils::get($this->api_key, "campaign/{$campaign_id}/stats");
    
    if (is_wp_error($response)) {
      // Cache errors for a short time
      set_transient($cache_key, $response, 5 * MINUTE_IN_SECONDS);
      return $response;
    }
    
    // Cache successful responses for 1 day
    set_transient($cache_key, $response, DAY_IN_SECONDS);
    
    return $response;
  }

  /**
   * Get campaign details
   *
   * @param string $campaign_id Campaign ID
   * @return array|WP_Error
   */
  public function get_campaign_details($campaign_id)
  {
    if (empty($campaign_id)) {
      return new WP_Error('invalid_campaign_id', __('Invalid campaign ID', 'campaigns-lite-for-laposta'));
    }

    // Check for a cached response - use a 30-day cache to reduce API calls
    // Since campaign details rarely change after sending
    $cache_key = 'laposta_campaign_details_' . $campaign_id;
    $cached_response = get_transient($cache_key);
    
    if ($cached_response !== false) {
      return $cached_response;
    }

    // Verify API key is available
    if (empty($this->api_key)) {
      return new WP_Error('no_api_key', __('API key is not set. Please select a valid API key.', 'campaigns-lite-for-laposta'));
    }
    
    // Make direct API call for campaign details
    $response = Campaigns_Lite_For_Laposta_API_Utils::get($this->api_key, "campaign/{$campaign_id}");
    
    if (is_wp_error($response)) {
      // Cache errors for a short time
      set_transient($cache_key, $response, 5 * MINUTE_IN_SECONDS);
      return $response;
    }
    
    // Process response to ensure it has the expected format
    $processed_response = $this->normalize_campaign_response($response);
    
    // Cache the processed response for 30 days - since campaign data rarely changes after sending
    set_transient($cache_key, $processed_response, 30 * DAY_IN_SECONDS);
    
    return $processed_response;
  }
  
  /**
   * Normalize a campaign response to ensure consistent format
   *
   * @param mixed $response The API response
   * @return array The normalized response
   */
  private function normalize_campaign_response($response)
  {
    if (!is_array($response)) {
      // If not an array, it might be some other type of object or data, so just return it
      return $response;
    } 
    
    if (isset($response['campaign'])) {
      // Already has campaign wrapper
      return $response;
    } 
    
    if (isset($response['data']) && is_array($response['data'])) {
      // Data property contains campaign info
      return array('campaign' => $response['data']);
    } 
    
    if (isset($response['campaign_id']) || isset($response['id'])) {
      // Direct campaign object
      return array('campaign' => $response);
    }
    
    // If we can't determine the format, return it as is
    return $response;
  }
}