<?php
/**
 * Plugin Name: Campaigns Lite for Laposta
 * Plugin URI: https://miix.dev/wp/campaigns-lite-for-laposta
 * Description: Lite version to display and manage your Laposta email campaigns within WordPress.
 * Version: 1.0.0
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author: Miix - Fred Klopper
 * Author URI: https://miix.dev
 * Text Domain: campaigns-lite-for-laposta
 * Domain Path: /languages
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

// Prevent direct access to this file
if (!defined('ABSPATH')) {
  exit;
}

// Enable error logging for debugging
if (!defined('WP_DEBUG_LOG')) {
  define('WP_DEBUG_LOG', true);
}

// Define plugin constants
define('CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION', '1.0.0');
define('CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_FILE', __FILE__);
define('CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Initialize plugin
 */
function campaigns_lite_for_laposta_init()
{

  // Include required files
  require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-settings.php';

  // Initialize settings
  if (is_admin()) {
    $settings = new Campaigns_Lite_For_Laposta_Settings();

    // Register AJAX handlers
    add_action('wp_ajax_campaigns_lite_for_laposta_fetch_campaign_details', array($settings, 'ajax_fetch_campaign_details'));
    add_action('wp_ajax_campaigns_lite_for_laposta_clear_campaign_cache', array($settings, 'ajax_clear_campaign_cache'));
  }

  // Register the background processing hook
  add_action('campaigns_lite_for_laposta_process_campaign_queue', 'campaigns_lite_for_laposta_process_queue');

  // Add admin notice for background processing
  add_action('admin_notices', 'campaigns_lite_for_laposta_background_notice');
}

/**
 * Display admin notice for background processing
 */
function campaigns_lite_for_laposta_background_notice()
{
  // Only show on our plugin pages or dashboard
  $screen = get_current_screen();
  $valid_screens = ['dashboard', 'toplevel_page_campaigns-lite-for-laposta-settings'];

  if (!$screen || !in_array($screen->id, $valid_screens)) {
    return;
  }

  // Only show on campaigns page if we're already there
  // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for page identification only, not processing form data
  $current_page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
  // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Used for tab identification only, not processing form data
  $current_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : '';
  $on_campaigns_page = ($current_page === 'campaigns-lite-for-laposta-settings' && $current_tab === 'campaigns');

  // Check if we have campaigns being processed
  $queue = get_transient('campaigns_lite_for_laposta_queue');
  $queue_count = $queue && is_array($queue) ? count($queue) : 0;

  if ($queue_count > 0 && !$on_campaigns_page) {
    ?>
    <div class="notice notice-info is-dismissible">
      <p>
        <span class="dashicons dashicons-update"
          style="animation: laposta-spin 2s infinite linear; vertical-align: middle; margin-right: 5px;"></span>
        <strong><?php esc_html_e('Campaigns Lite for Laposta:', 'campaigns-lite-for-laposta'); ?></strong>
        <?php printf(
          // Translators: %1$d is the number of campaigns being processed
          esc_html__('%1$d campaigns are being processed in the background. This happens gradually to avoid API rate limits.', 'campaigns-lite-for-laposta'),
          intval($queue_count)
        ); ?>
        <a href="<?php echo esc_url(admin_url('admin.php?page=campaigns-lite-for-laposta-settings&tab=campaigns')); ?>"
          class="button button-small" style="margin-left: 10px; vertical-align: middle;">
          <?php esc_html_e('View Campaigns', 'campaigns-lite-for-laposta'); ?>
        </a>
        <a href="#" class="refresh-status button button-small" style="margin-left: 5px; vertical-align: middle;"
          onclick="location.reload(); return false;">
          <?php esc_html_e('Refresh Status', 'campaigns-lite-for-laposta'); ?>
        </a>
      </p>
    </div>
    <style>
      @keyframes laposta-spin {
        0% {
          transform: rotate(0deg);
        }

        100% {
          transform: rotate(360deg);
        }
      }
    </style>
    <?php
  }
}

/**
 * Background process to fetch campaign details
 *
 * Note: This function is now simplified since we get all data in a single API call.
 * It's kept for compatibility and potential future use.
 */
function campaigns_lite_for_laposta_process_queue()
{
  // Clear the queue since we don't need it anymore - we get everything in a single API call
  delete_transient('campaigns_lite_for_laposta_queue');

  // Make sure API is initialized
  require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-api.php';
  $api = new Campaigns_Lite_For_Laposta_API(false);

  // Just do a fresh get_campaigns() call which will cache the result
  $api->get_campaigns();

  // Refreshed all campaign data with a single API call
}
add_action('plugins_loaded', 'campaigns_lite_for_laposta_init');

/**
 * Activation hook
 */
function campaigns_lite_for_laposta_activate()
{
  // Add default options
  add_option('campaigns_lite_for_laposta_api_key', '');

  // Clear permalinks
  flush_rewrite_rules();
}
register_activation_hook(CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_FILE, 'campaigns_lite_for_laposta_activate');

/**
 * Deactivation hook
 */
function campaigns_lite_for_laposta_deactivate()
{
  // Clear permalinks
  flush_rewrite_rules();
}
register_deactivation_hook(CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_FILE, 'campaigns_lite_for_laposta_deactivate');


/**
 * Register shortcode
 */
function campaigns_lite_for_laposta_register_shortcode()
{
  add_shortcode('campaigns_lite_for_laposta', 'campaigns_lite_for_laposta_shortcode');
}
add_action('init', 'campaigns_lite_for_laposta_register_shortcode');

/**
 * Shortcode handler
 *
 * @param array $atts Shortcode attributes
 * @return string HTML output
 */
function campaigns_lite_for_laposta_shortcode($atts)
{
  // Enqueue styles
  wp_enqueue_style(
    'laposta-campaigns-lite-shortcode',
    CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/css/campaigns-lite-for-laposta-shortcode.css',
    array(),
    CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION
  );

  // Parse shortcode attributes
  $atts = shortcode_atts(array(
    'number' => 3,
    'show_screenshot' => 'yes',
    'show_date' => 'yes'
  ), $atts);

  // LITE VERSION: Limit to maximum 12 campaigns
  $number = absint($atts['number']);
  if ($number < 1 || $number > 12) {
    $number = 3; // Default to 3 if not specified or invalid
  }

  // Convert string boolean attributes to actual booleans
  $show_screenshot = filter_var($atts['show_screenshot'], FILTER_VALIDATE_BOOLEAN);
  $show_date = filter_var($atts['show_date'], FILTER_VALIDATE_BOOLEAN);

  // Use WordPress default date format
  $date_format = get_option('date_format');

  // Lite version: Fixed simple grid layout (3 columns)
  $layout = 'grid';
  $columns = 3;

  // Initialize API
  require_once CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_DIR . 'includes/class-campaigns-lite-for-laposta-api.php';
  $api = new Campaigns_Lite_For_Laposta_API();

  // Get selected API key
  if (class_exists('Campaigns_Lite_For_Laposta_Settings')) {
    $api_key = Campaigns_Lite_For_Laposta_Settings::get_actual_api_key();
  } else {
    $api_key = get_option('campaigns_lite_for_laposta_api_key');
  }

  if (empty($api_key)) {
    return '<p class="laposta-campaigns-error">' . esc_html__('No API key selected. Please configure the Campaigns Lite for Laposta plugin.', 'campaigns-lite-for-laposta') . '</p>';
  }

  // Get campaigns from API
  $response = $api->get_campaigns();

  if (is_wp_error($response)) {
    return '<p class="laposta-campaigns-error">' . esc_html__('Error retrieving campaigns. Please check the error logs.', 'campaigns-lite-for-laposta') . '</p>';
  }

  if (empty($response)) {
    // Empty response from API
    return '<p class="laposta-campaigns-error">' . esc_html__('No campaigns returned from API.', 'campaigns-lite-for-laposta') . '</p>';
  }

  // Log what we got from the API
  // Got campaigns from API

  // First, normalize the response format
  $normalized_response = $response;

  // Check if we have a direct array of campaigns without the campaign wrapper
  if (isset($normalized_response[0]) && !isset($normalized_response[0]['campaign']) && isset($normalized_response[0]['campaign_id'])) {
    $fixed_response = array();
    foreach ($normalized_response as $campaign) {
      $fixed_response[] = array('campaign' => $campaign);
    }
    $normalized_response = $fixed_response;
    // Normalized response format
  }

  // Find all sent campaigns first
  $all_sent_campaigns = array();
  foreach ($normalized_response as $item) {
    if (!isset($item['campaign'])) {
      continue;
    }

    if (!empty($item['campaign']['delivery_ended'])) {
      $all_sent_campaigns[] = $item;
      // Found sent campaign
    }
  }

  if (empty($all_sent_campaigns)) {
    // No sent campaigns found
    return '<p class="laposta-campaigns-error">' . esc_html__('No sent campaigns found. Only sent campaigns are displayed.', 'campaigns-lite-for-laposta') . '</p>';
  }

  // LITE VERSION: No filtering - just use all sent campaigns
  $sent_campaigns = $all_sent_campaigns;

  // Sort by delivery date (newest first)
  usort($sent_campaigns, function ($a, $b) {
    $date_a = strtotime($a['campaign']['delivery_ended']);
    $date_b = strtotime($b['campaign']['delivery_ended']);
    return $date_b - $date_a;
  });

  // Limit campaigns to specified number
  $sent_campaigns = array_slice($sent_campaigns, 0, $number);

  // Now enrich each campaign with details (screenshot and web URL)
  foreach ($sent_campaigns as $key => $item) {
    if (empty($item['campaign']['campaign_id'])) {
      continue;
    }

    $campaign_id = $item['campaign']['campaign_id'];
    // Enriching campaign

    // Only fetch details if the campaign is missing screenshot or web URL
    if (empty($item['campaign']['screenshot']) || empty($item['campaign']['web'])) {
      // Fetch details via API - with built-in caching
      $details = $api->get_campaign_details($campaign_id);

      if (!is_wp_error($details) && isset($details['campaign'])) {
        // Update screenshot if provided
        if (!empty($details['campaign']['screenshot'])) {
          $sent_campaigns[$key]['campaign']['screenshot'] = $details['campaign']['screenshot'];
          // Added screenshot to campaign
        }

        // Update web URL if provided
        if (!empty($details['campaign']['web'])) {
          $sent_campaigns[$key]['campaign']['web'] = $details['campaign']['web'];
          // Added web URL to campaign
        }
      } else if (is_wp_error($details)) {
        // Error fetching details for campaign
      }
    }
  }

  // Create cache key based on API key and shortcode parameters for caching enriched results
  $cache_key = 'laposta_shortcode_' . md5($api_key . serialize($atts));
  
  // Cache the results for 1 hour
  set_transient($cache_key, $sent_campaigns, HOUR_IN_SECONDS);

  // Final campaign count

  // Debugging shortcode output
  if (current_user_can('manage_options')) {
    // Debug output for admin
  }

  ob_start();

  // LITE VERSION: Simple fixed grid layout (3 columns)
  $container_classes = array('laposta-campaigns-list-shortcode');
  $container_classes[] = 'laposta-campaigns-layout-grid';
  $container_classes[] = 'laposta-campaigns-columns-3';
  
  ?>
  <div class="<?php echo esc_attr(implode(' ', $container_classes)); ?>">
    <?php
    if (empty($sent_campaigns)) {
      // Provide more detailed debug information for admin users
      if (current_user_can('manage_options')) {
        echo '<div class="laposta-campaigns-admin-debug">';
        echo '<p><strong>' . esc_html__('Admin Debug Info:', 'campaigns-lite-for-laposta') . '</strong></p>';
        echo '<p>' . esc_html__('No campaigns were found to display. This could be due to:', 'campaigns-lite-for-laposta') . '</p>';
        echo '<ul>';
        echo '<li>' . esc_html__('No campaigns exist in your Laposta account', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('No campaigns have been sent yet (only sent campaigns are displayed)', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('The API key does not have access to campaigns', 'campaigns-lite-for-laposta') . '</li>';
        echo '<li>' . esc_html__('There was an error enriching campaign data', 'campaigns-lite-for-laposta') . '</li>';
        echo '</ul>';

        echo '<p>' . esc_html__('API Configuration:', 'campaigns-lite-for-laposta') . '</p>';
        echo '<ul>';
        echo '<li>' . esc_html__('API Key Set:', 'campaigns-lite-for-laposta') . ' ' . (!empty($api_key) ? esc_html__('Yes', 'campaigns-lite-for-laposta') : esc_html__('No', 'campaigns-lite-for-laposta')) . '</li>';
        echo '<li>' . esc_html__('API Response Count:', 'campaigns-lite-for-laposta') . ' ' . (is_array($response) ? esc_html(count($response)) : esc_html__('Not an array', 'campaigns-lite-for-laposta')) . '</li>';
        echo '<li>' . esc_html__('Sent Campaigns Found:', 'campaigns-lite-for-laposta') . ' ' . esc_html(count($all_sent_campaigns)) . '</li>';
        echo '</ul>';
        echo '</div>';
      }

      echo '<p class="laposta-campaigns-empty">' . esc_html__('No campaigns where found to display.', 'campaigns-lite-for-laposta') . '</p>';
    } else {
      foreach ($sent_campaigns as $item):
        if (!isset($item['campaign'])) {
          continue;
        }

        $campaign = $item['campaign'];

        // Skip if we don't have the required data
        if (empty($campaign['web'])) {
          if (current_user_can('manage_options')) {
            echo '<div class="laposta-campaigns-admin-debug">';
            echo '<p>' . esc_html__('Campaign missing web URL:', 'campaigns-lite-for-laposta') . ' ';
            echo !empty($campaign['name']) ? esc_html($campaign['name']) : (!empty($campaign['campaign_id']) ? esc_html($campaign['campaign_id']) : esc_html__('Unknown', 'campaigns-lite-for-laposta'));
            echo '</p>';
            echo '</div>';
          }
          continue;
        }

        // Determine the campaign title - prioritize subject (public title) over name (internal name)
        $campaign_name = '';
        if (!empty($campaign['subject'])) {
          $campaign_name = sanitize_text_field($campaign['subject']);
        } else if (!empty($campaign['title'])) {
          $campaign_name = sanitize_text_field($campaign['title']);
        } else if (!empty($campaign['name'])) {
          $campaign_name = sanitize_text_field($campaign['name']);
        } else {
          $campaign_name = esc_html__('Untitled Campaign', 'campaigns-lite-for-laposta');
        }
        
        // LITE VERSION: Simple fixed item classes
        ?>
        <div class="laposta-campaigns-item">
          <div class="laposta-campaigns-content">
            <?php if ($show_date && !empty($campaign['delivery_ended'])): ?>
              <div class="laposta-campaigns-date">
                <?php
                $date = strtotime($campaign['delivery_ended']);
                echo esc_html(date_i18n($date_format, $date));
                ?>
              </div>
            <?php endif; ?>
            <h3 class="laposta-campaigns-title">
              <a href="<?php echo esc_url($campaign['web']); ?>" target="_blank">
                <?php echo esc_html($campaign_name); ?>
              </a>
            </h3>
            <?php
            // Show screenshot if available in any size
            if ($show_screenshot):
              $screenshot_url = '';

              // Try to get the best size for display
              if (!empty($campaign['screenshot']) && isset($campaign['screenshot']['226x268'])) {
                $screenshot_url = $campaign['screenshot']['226x268'];
              } elseif (!empty($campaign['screenshot']) && is_array($campaign['screenshot'])) {
                // Just grab the first available size
                $screenshot_url = reset($campaign['screenshot']);
              }

              if (!empty($screenshot_url)):
                ?>
                <div class="laposta-campaigns-screenshot">
                  <a href="<?php echo esc_url($campaign['web']); ?>" target="_blank">
                    <img src="<?php echo esc_url($screenshot_url); ?>" alt="<?php echo esc_attr($campaign_name); ?>" />
                  </a>
                </div>
                <?php
              elseif (current_user_can('manage_options')):
                echo '<div class="laposta-campaigns-admin-debug">';
                echo '<p>' . esc_html__('Campaign missing screenshot:', 'campaigns-lite-for-laposta') . ' ' . esc_html($campaign_name) . '</p>';
                echo '</div>';
              endif;
            endif;
            ?>
          </div>
        </div>
      <?php endforeach;
    }
    ?>
  </div>
  <?php
  return ob_get_clean();
}

/**
 * Register and enqueue shortcode styles
 */
function campaigns_lite_for_laposta_register_shortcode_styles()
{
  if (!is_admin()) {
    wp_enqueue_style(
      'laposta-campaigns-lite-shortcode',
      CAMPAIGNS_LITE_FOR_LAPOSTA_PLUGIN_URL . 'assets/css/campaigns-lite-for-laposta-shortcode.css',
      array(),
      CAMPAIGNS_LITE_FOR_LAPOSTA_VERSION
    );
  }
}

/**
 * Add settings link on plugin page
 */
function campaigns_lite_for_laposta_plugin_links($links)
{
  $settings_link = '<a href="admin.php?page=campaigns-lite-for-laposta-settings">' . __('Settings', 'campaigns-lite-for-laposta') . '</a>';
  array_unshift($links, $settings_link);
  return $links;
}
$plugin = plugin_basename(__FILE__);
add_filter("plugin_action_links_$plugin", 'campaigns_lite_for_laposta_plugin_links');

// Customize plugin row meta links
add_filter('plugin_row_meta', 'campaigns_lite_for_laposta_plugin_row_meta', 10, 2);

/**
 * Customize plugin row meta links
 *
 * @param array $links Array of plugin row meta links
 * @param string $file Plugin file path
 * @return array Modified links array
 */
function campaigns_lite_for_laposta_plugin_row_meta($links, $file) {
    if (plugin_basename(__FILE__) === $file) {
        // Remove default 'View details' link if present
        if (isset($links[2])) {
            unset($links[2]);
        }

        // Add custom 'Visit the plugin site' link
        $links[] = '<a href="https://miix.dev/wp/campaigns-lite-for-laposta" target="_blank">' . __('Visit the plugin site', 'campaigns-lite-for-laposta') . '</a>';
    }
    return $links;
}

/**
 * Parse max_age parameter and return timestamp limit
 *
 * @param string $max_age The max age parameter (e.g., "30", "3m", "1y")
 * @return int|false Timestamp limit or false if invalid
 */
function campaigns_lite_for_laposta_parse_max_age($max_age) {
  if (empty($max_age)) {
    return false;
  }
  
  $max_age = trim($max_age);
  $current_time = time();
  
  // Check if it's just a number (days)
  if (is_numeric($max_age)) {
    $days = intval($max_age);
    if ($days > 0) {
      return $current_time - ($days * 24 * 60 * 60);
    }
  }
  
  // Check for month/year suffixes
  if (preg_match('/^(\d+)([myd])$/i', $max_age, $matches)) {
    $number = intval($matches[1]);
    $unit = strtolower($matches[2]);
    
    if ($number > 0) {
      switch ($unit) {
        case 'd': // days
          return $current_time - ($number * 24 * 60 * 60);
        case 'm': // months (approximate 30 days)
          return $current_time - ($number * 30 * 24 * 60 * 60);
        case 'y': // years (approximate 365 days)
          return $current_time - ($number * 365 * 24 * 60 * 60);
      }
    }
  }
  
  return false;
}

/**
 * Render campaigns in list format (ul, ol, or div containers)
 *
 * @param array $campaigns Array of campaign data
 * @param string $list_format List format: 'ul', 'ol', or '' (div)
 * @param string $container_class Custom container classes
 * @param string $item_class Custom item classes  
 * @param bool $show_date Whether to show dates
 * @param string $date_format Date format string
 * @param bool $custom_css_only Whether to use only custom CSS classes
 * @return string HTML output
 */
function campaigns_lite_for_laposta_render_list_format($campaigns, $list_format, $container_class, $item_class, $show_date, $date_format, $custom_css_only = false) {
  if (empty($campaigns)) {
    return '<p class="laposta-campaigns-empty">' . esc_html__('No campaigns where found to display.', 'campaigns-lite-for-laposta') . '</p>';
  }
  
  // Build container classes
  $container_classes = array();
  if ($custom_css_only) {
    // Use only custom classes when custom_css_only is enabled
    if (!empty($container_class)) {
      $custom_classes = explode(' ', $container_class);
      foreach ($custom_classes as $class) {
        $class = trim($class);
        if (!empty($class)) {
          $container_classes[] = sanitize_html_class($class);
        }
      }
    }
  } else {
    // Default behavior: use plugin classes + optional custom classes
    $container_classes[] = 'laposta-campaigns-list';
    if (!empty($list_format)) {
      $container_classes[] = 'laposta-campaigns-' . $list_format;
    }
    if (!empty($container_class)) {
      $custom_classes = explode(' ', $container_class);
      foreach ($custom_classes as $class) {
        $class = trim($class);
        if (!empty($class)) {
          $container_classes[] = sanitize_html_class($class);
        }
      }
    }
  }
  
  // Build item classes
  $item_classes = array();
  if ($custom_css_only) {
    // Use only custom item class when custom_css_only is enabled
    if (!empty($item_class)) {
      $item_classes[] = sanitize_html_class($item_class);
    }
  } else {
    // Default behavior: use plugin classes + optional custom item class
    $item_classes[] = 'laposta-campaign-item';
    if (!empty($item_class)) {
      $item_classes[] = sanitize_html_class($item_class);
    }
  }
  
  $output = '';
  
  // Start wrapper
  if ($list_format === 'ul' || $list_format === 'ol') {
    $tag = $list_format;
    $item_tag = 'li';
  } else {
    $tag = 'div';
    $item_tag = 'div';
  }
  
  $class_attr = !empty($container_classes) ? ' class="' . esc_attr(implode(' ', $container_classes)) . '"' : '';
  $output .= '<' . $tag . $class_attr . '>';
  
  // Process each campaign
  foreach ($campaigns as $item) {
    if (!isset($item['campaign']) || empty($item['campaign']['web'])) {
      continue;
    }
    
    $campaign = $item['campaign'];
    
    // Determine the campaign title
    $campaign_name = '';
    if (!empty($campaign['subject'])) {
      $campaign_name = sanitize_text_field($campaign['subject']);
    } else if (!empty($campaign['title'])) {
      $campaign_name = sanitize_text_field($campaign['title']);
    } else if (!empty($campaign['name'])) {
      $campaign_name = sanitize_text_field($campaign['name']);
    } else {
      $campaign_name = esc_html__('Untitled Campaign', 'campaigns-lite-for-laposta');
    }
    
    // Build item
    $item_class_attr = !empty($item_classes) ? ' class="' . esc_attr(implode(' ', $item_classes)) . '"' : '';
    $output .= '<' . $item_tag . $item_class_attr . '>';
    
    // Add link
    $output .= '<a href="' . esc_url($campaign['web']) . '" target="_blank">' . esc_html($campaign_name) . '</a>';
    
    // Add date if requested
    if ($show_date && !empty($campaign['delivery_ended'])) {
      $date = strtotime($campaign['delivery_ended']);
      $output .= ' <span class="laposta-campaign-date">| ' . esc_html(date_i18n($date_format, $date)) . '</span>';
    }
    
    $output .= '</' . $item_tag . '>';
  }
  
  $output .= '</' . $tag . '>';
  
  return $output;
}